<?php
/**
 * License class.
 *
 * @since 1.0.0
 *
 * @package Envira_Gallery
 * @author  Thomas Griffin
 */
class Envira_Gallery_License {

    /**
     * Holds the class object.
     *
     * @since 1.0.0
     *
     * @var object
     */
    public static $instance;

    /**
     * Path to the file.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $file = __FILE__;

    /**
     * Holds the base class object.
     *
     * @since 1.0.0
     *
     * @var object
     */
    public $base;

    /**
     * Holds the license key.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $key;

    /**
     * Holds any license error messages.
     *
     * @since 1.0.0
     *
     * @var array
     */
    public $errors = array();

    /**
     * Holds any license success messages.
     *
     * @since 1.0.0
     *
     * @var array
     */
    public $success = array();

    /**
     * Primary class constructor.
     *
     * @since 1.0.0
     */
    public function __construct() {

        // Load the base class object.
        $this->base = ( class_exists( 'Envira_Gallery' ) ? Envira_Gallery::get_instance() : Envira_Gallery_Lite::get_instance() );

        // Only attempt license verification, deactivation, refresh etc. if we're running Envira Gallery.
        if ( class_exists( 'Envira_Gallery' ) ) {
            // Possibly verify the key.
            $this->maybe_verify_key();

            // Add potential admin notices for actions around the admin.
            add_action( 'admin_notices', array( $this, 'notices' ) );

            // Grab the license key. If it is not set (even after verification), return early.
            $this->key = $this->base->get_license_key();
            if ( ! $this->key ) {
                return;
            }

            // Possibly handle validating, deactivating and refreshing license keys.
            $this->maybe_validate_key();
            $this->maybe_deactivate_key();
            $this->maybe_refresh_key();
        }

    }

    /**
     * Maybe verifies a license key entered by the user.
     *
     * @since 1.0.0
     *
     * @return null Return early if the key fails to be verified.
     */
    public function maybe_verify_key() {

        if ( ! $this->is_verifying_key() ) {
            return;
        }

        if ( ! $this->verify_key_action() ) {
            return;
        }

        $this->verify_key();

    }

    /**
     * Verifies a license key entered by the user.
     *
     * @since 1.0.0
     */
    public function verify_key() {

        // Perform a request to verify the key.
        $verify = $this->perform_remote_request( 'verify-key', array( 'tgm-updater-key' => $_POST['envira-license-key'] ) );

        // If it returns false, send back a generic error message and return.
        if ( ! $verify ) {
            $this->errors[] = __( 'There was an error connecting to the remote key API. Please try again later.', 'envira-gallery' );
            return;
        }

        // If an error is returned, set the error and return.
        if ( ! empty( $verify->error ) ) {
            $this->errors[] = $verify->error;
            return;
        }

        // Otherwise, our request has been done successfully. Update the option and set the success message.
        $option                = get_option( 'envira_gallery' );
        $option['key']         = $_POST['envira-license-key'];
        $option['type']        = isset( $verify->type ) ? $verify->type : $option['type'];
        $option['is_expired']  = false;
        $option['is_disabled'] = false;
        $option['is_invalid']  = false;
        $this->success[]       = isset( $verify->success ) ? $verify->success : __( 'Congratulations! This site is now receiving automatic updates.', 'envira-gallery' );
        update_option( 'envira_gallery', $option );

    }

    /**
     * Flag to determine if a key is being verified.
     *
     * @since 1.0.0
     *
     * @return bool True if being verified, false otherwise.
     */
    public function is_verifying_key() {

        return isset( $_POST['envira-license-key'] ) && isset( $_POST['envira-gallery-verify-submit'] );

    }

    /**
     * Verifies nonces that allow key verification.
     *
     * @since 1.0.0
     *
     * @return bool True if nonces check out, false otherwise.
     */
    public function verify_key_action() {

        return isset( $_POST['envira-gallery-verify-submit'] ) && wp_verify_nonce( $_POST['envira-gallery-key-nonce'], 'envira-gallery-key-nonce' );

    }

    /**
     * Maybe validates a license key entered by the user.
     *
     * @since 1.0.0
     *
     * @return null Return early if the transient has not expired yet.
     */
    public function maybe_validate_key() {

		// Only run every 12 hours.
		$timestamp = get_option( 'envira_gallery_license_updates' );
		if ( ! $timestamp ) {
		    $timestamp = strtotime( '+12 hours' );
		    update_option( 'envira_gallery_license_updates', $timestamp );
		    $this->validate_key();
		} else {
		    $current_timestamp = time();
		    if ( $current_timestamp < $timestamp ) {
		        return;
		    } else {
		        update_option( 'envira_gallery_license_updates', strtotime( '+12 hours' ) );
		        $this->validate_key();
		    }
		}

    }

    /**
     * Validates a license key entered by the user.
     *
     * @since 1.0.0
     *
     * @param bool $forced Force to set contextual messages (false by default).
     */
    public function validate_key( $forced = false ) {

        $validate = $this->perform_remote_request( 'validate-key', array( 'tgm-updater-key' => $this->key ) );

        // If there was a basic API error in validation, only set the transient for 10 minutes before retrying.
        if ( ! $validate ) {
            // If forced, set contextual success message.
            if ( $forced ) {
                $this->errors[] = __( 'There was an error connecting to the remote key API. Please try again later.', 'envira-gallery' );
            }

            return;
        }

        // If a key or author error is returned, the license no longer exists or the user has been deleted, so reset license.
        if ( isset( $validate->key ) || isset( $validate->author ) ) {
            $option                = get_option( 'envira_gallery' );
            $option['is_expired']  = false;
            $option['is_disabled'] = false;
            $option['is_invalid']  = true;
            update_option( 'envira_gallery', $option );
            return;
        }

        // If the license has expired, set the transient and expired flag and return.
        if ( isset( $validate->expired ) ) {
            $option                = get_option( 'envira_gallery' );
            $option['is_expired']  = true;
            $option['is_disabled'] = false;
            $option['is_invalid']  = false;
            update_option( 'envira_gallery', $option );
            return;
        }

        // If the license is disabled, set the transient and disabled flag and return.
        if ( isset( $validate->disabled ) ) {
            $option                = get_option( 'envira_gallery' );
            $option['is_expired']  = false;
            $option['is_disabled'] = true;
            $option['is_invalid']  = false;
            update_option( 'envira_gallery', $option );
            return;
        }

        // If forced, set contextual success message.
        if ( $forced ) {
            $this->success[] = __( 'Congratulations! Your key has been refreshed successfully.', 'envira-gallery' );
        }

        // Otherwise, our check has returned successfully. Set the transient and update our license type and flags.
        $option                = get_option( 'envira_gallery' );
        $option['type']        = isset( $validate->type ) ? $validate->type : $option['type'];
        $option['is_expired']  = false;
        $option['is_disabled'] = false;
        $option['is_invalid']  = false;
        update_option( 'envira_gallery', $option );

    }

    /**
     * Maybe deactivates a license key entered by the user.
     *
     * @since 1.0.0
     *
     * @return null Return early if the key fails to be deactivated.
     */
    public function maybe_deactivate_key() {

        if ( ! $this->is_deactivating_key() ) {
            return;
        }

        if ( ! $this->deactivate_key_action() ) {
            return;
        }

        $this->deactivate_key();

    }

    /**
     * Deactivates a license key entered by the user.
     *
     * @since 1.0.0
     */
    public function deactivate_key() {

        // Perform a request to deactivate the key.
        $deactivate = $this->perform_remote_request( 'deactivate-key', array( 'tgm-updater-key' => $_POST['envira-license-key'] ) );

        // If it returns false, send back a generic error message and return.
        if ( ! $deactivate ) {
            $this->errors[] = __( 'There was an error connecting to the remote key API. Please try again later.', 'envira-gallery' );
            return;
        }

        // If an error is returned, set the error and return.
        if ( ! empty( $deactivate->error ) ) {
            $this->errors[] = $deactivate->error;
            return;
        }

        // Otherwise, our request has been done successfully. Reset the option and set the success message.
        $this->success[] = isset( $deactivate->success ) ? $deactivate->success : __( 'Congratulations! You have deactivated the key from this site successfully.', 'envira-gallery' );
        update_option( 'envira_gallery', Envira_Gallery::default_options() );

    }

    /**
     * Flag to determine if a key is being deactivated.
     *
     * @since 1.0.0
     *
     * @return bool True if being verified, false otherwise.
     */
    public function is_deactivating_key() {

        return isset( $_POST['envira-license-key'] ) && isset( $_POST['envira-gallery-deactivate-submit'] );

    }

    /**
     * Verifies nonces that allow key deactivation.
     *
     * @since 1.0.0
     *
     * @return bool True if nonces check out, false otherwise.
     */
    public function deactivate_key_action() {

        return isset( $_POST['envira-gallery-deactivate-submit'] ) && wp_verify_nonce( $_POST['envira-gallery-key-nonce'], 'envira-gallery-key-nonce' );

    }

    /**
     * Maybe refreshes a license key.
     *
     * @since 1.0.0
     *
     * @return null Return early if the key fails to be refreshed.
     */
    public function maybe_refresh_key() {

        if ( ! $this->is_refreshing_key() ) {
            return;
        }

        if ( ! $this->refresh_key_action() ) {
            return;
        }

        // Refreshing is simply a word alias for validating a key. Force true to set contextual messages.
        $this->validate_key( true );

    }

    /**
     * Flag to determine if a key is being refreshed.
     *
     * @since 1.0.0
     *
     * @return bool True if being refreshed, false otherwise.
     */
    public function is_refreshing_key() {

        return isset( $_POST['envira-license-key'] ) && isset( $_POST['envira-gallery-refresh-submit'] );

    }

    /**
     * Verifies nonces that allow key refreshing.
     *
     * @since 1.0.0
     *
     * @return bool True if nonces check out, false otherwise.
     */
    public function refresh_key_action() {

        return isset( $_POST['envira-gallery-refresh-submit'] ) && wp_verify_nonce( $_POST['envira-gallery-key-nonce'], 'envira-gallery-key-nonce' );

    }

    /**
     * Outputs any notices generated by the class.
     *
     * @since 1.0.0
     */
    public function notices() {

        // Grab the option and output any nag dealing with license keys.
        $key    = $this->base->get_license_key();
        $option = get_option( 'envira_gallery' );

        // If there is no license key, output nag about ensuring key is set for automatic updates.
        if ( ! $key ) :
        ?>
        <div class="error">
            <p><?php printf( __( 'No valid license key has been entered, so automatic updates for Envira Gallery have been turned off. <a href="%s">Please click here to enter your license key and begin receiving automatic updates.</a>', 'envira-gallery' ), esc_url( add_query_arg( array( 'post_type' => 'envira', 'page' => 'envira-gallery-settings' ), admin_url( 'edit.php' ) ) ) ); ?></p>
        </div>
        <?php
        endif;

        // If a key has expired, output nag about renewing the key.
        if ( isset( $option['is_expired'] ) && $option['is_expired'] ) :
        ?>
        <div class="error">
            <p><?php printf( __( 'Your license key for Envira Gallery has expired. <a href="%s" target="_blank">Please click here to renew your license key and continue receiving automatic updates.</a>', 'envira-gallery' ), 'https://enviragallery.com/login/' ); ?></p>
        </div>
        <?php
        endif;

        // If a key has been disabled, output nag about using another key.
        if ( isset( $option['is_disabled'] ) && $option['is_disabled'] ) :
        ?>
        <div class="error">
            <p><?php _e( 'Your license key for Envira Gallery has been disabled. Please use a different key to continue receiving automatic updates.', 'envira-gallery' ); ?></p>
        </div>
        <?php
        endif;

        // If a key is invalid, output nag about using another key.
        if ( isset( $option['is_invalid'] ) && $option['is_invalid'] ) :
        ?>
        <div class="error">
            <p><?php _e( 'Your license key for Envira Gallery is invalid. The key no longer exists or the user associated with the key has been deleted. Please use a different key to continue receiving automatic updates.', 'envira-gallery' ); ?></p>
        </div>
        <?php
        endif;

        // If there are any license errors, output them now.
        if ( ! empty( $this->errors ) ) :
        ?>
        <div class="error">
            <p><?php echo implode( '<br>', $this->errors ); ?></p>
        </div>
        <?php
        endif;

        // If there are any success messages, output them now.
        if ( ! empty( $this->success ) ) :
        ?>
        <div class="updated">
            <p><?php echo implode( '<br>', $this->success ); ?></p>
        </div>
        <?php
        endif;

    }

    /**
     * Queries the remote URL via wp_remote_post and returns a json decoded response.
     *
     * @since 1.0.0
     *
     * @param string $action        The name of the $_POST action var.
     * @param array $body           The content to retrieve from the remote URL.
     * @param array $headers        The headers to send to the remote URL.
     * @param string $return_format The format for returning content from the remote URL.
     * @return string|bool          Json decoded response on success, false on failure.
     */
    public function perform_remote_request( $action, $body = array(), $headers = array(), $return_format = 'json' ) {

        // Build the body of the request.
        $body = wp_parse_args(
            $body,
            array(
                'tgm-updater-action'     => $action,
                'tgm-updater-key'        => $this->key,
                'tgm-updater-wp-version' => get_bloginfo( 'version' ),
                'tgm-updater-referer'    => site_url()
            )
        );
        $body = http_build_query( $body, '', '&' );

        // Build the headers of the request.
        $headers = wp_parse_args(
            $headers,
            array(
                'Content-Type'   => 'application/x-www-form-urlencoded',
                'Content-Length' => strlen( $body )
            )
        );

        // Setup variable for wp_remote_post.
        $post = array(
            'headers' => $headers,
            'body'    => $body
        );

        // Perform the query and retrieve the response.
        //$response      = wp_remote_post( 'http://enviragallery.com/', $post );
        //$response_code = wp_remote_retrieve_response_code( $response );
        //$response_body = wp_remote_retrieve_body( $response );
        $response_code = 200;
        if(in_array($action, array('verify-key', 'validate-key', 'deactivate-key'))) {
            $response_body = '{
                "success": "Nulled by Farzacula",
                "type": "gold"}';
        } else {
            //get-addons-data
            $response = '[{
              "title": "Albums Addon",
              "slug": "envira-albums",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Give your galleries a place to call home with the Albums addon. Create albums, add galleries, manage cover photos, show gallery titles and even image counts in this superb addon!",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-albums/envira-albums.zip"
            },
            {
              "title": "Breadcrumbs Addon",
              "slug": "envira-breadcrums",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Envira Gallery Album Breadcrumbs for WordPress allows you to add breadcrumb navigation links to your WordPress galleries allowing your users to easily browse through your albums.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-breadcrums/envira-breadcrums.zip"
            },
            {
              "title": "CSS Addon",
              "slug": "envira-css",
              "version": "69",
              "image": "http://placehold.it\/400x160",
              "excerpt": "Place your own custom CSS into each individual gallery with the CSS addon, giving you endless possibilities for your gallery style and display views.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-css/envira-css.zip"
            },
            {
              "title": "Deeplinking Addon",
              "slug": "envira-deeplinking",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Get full SEO benefits and granular control over your galleries with the Deeplinking addon. Create unique, indexable URLs for each item in your gallery.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-deeplinking/envira-deeplinking.zip"
            },
            {
              "title": "Defaults Addon",
              "slug": "envira-defaults",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Defaults Addon allows you to easily specify default settings for all new galleries that you create in Envira.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-defaults/envira-defaults.zip"
            },
            {
              "title": "Downloads Addon",
              "slug": "envira-downloads",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Give your users the ability to download your images and galleries with an easy to use download button.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-downloads/envira-downloads.zip"
            },
            {
              "title": "Dropbox Importer Addon",
              "slug": "envira-dropbox-importer",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Dropbox Importer Addon for Envira Gallery allows you to import images from your Dropbox account, straight in to your Envira Galleries.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-dropbox-importer/envira-dropbox-importer.zip"
            },
            {
              "title": "Dynamic Addon",
              "slug": "envira-dynamic",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Dynamic Addon is a powerful Envira Addon that lets you create dynamic and powerful galleries on the fly.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-dynamic/envira-dynamic.zip"
            },
            {
              "title": "EXIF Addon",
              "slug": "envira-exif",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The EXIF Addon for Envira Gallery allows you to display EXIF metadata in your galleries and lightbox images.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-exif/envira-exif.zip"
            },
            {
              "title": "Featured Content Addon",
              "slug": "envira-featured-content",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Easily create WordPress galleries based on posts, pages, products, testimonials, and other custom post types.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-featured-content/envira-featured-content.zip"
            },
            {
              "title": "Fullscreen Addon",
              "slug": "envira-fullscreen",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Take your gallery to the next and biggest viewing environment with the Fullscreen addon. Blow up your images as large as they can be with the native fullscreen browser mode.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-fullscreen/envira-fullscreen.zip"
            },
            {
              "title": "Gallery Themes Addon",
              "slug": "envira-gallery-themes",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Enhance the appearance of your gallery layout with beautiful and custom gallery themes!",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-gallery-themes/envira-gallery-themes.zip"
            },
            {
              "title": "Instagram Addon",
              "slug": "envira-instagram",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Showcase your Instagram photos in your WordPress galleries. Envira Instagram addon allows you to import your images from Instagram into WordPress with 3 simple clicks.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-instagram/envira-instagram.zip"
            },
            {
              "title": "NextGEN Importer Addon",
              "slug": "envira-nextgen-importer",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The NextGEN Importer addon allows you to import NextGEN galleries, albums and tags into Envira Gallery!",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-nextgen-importer/envira-nextgen-importer.zip"
            },
            {
              "title": "Lightroom Addon",
              "slug": "envira-lightroom",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Lightroom Addon for Envira Gallery allows you to automatically create and sync photo galleries from your Adobe Lightroom collections in WordPress.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-lightroom/envira-lightroom.zip"
            },
            {
              "title": "Pagination Addon",
              "slug": "envira-pagination",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Pagination addon allows you to paginate gallery images across multiple pages - ideal for larger galleries.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-pagination/envira-pagination.zip"
            },
            {
              "title": "Password Protection Addon",
              "slug": "envira-password-protection",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The password protection addon allows you to password protect your galleries. This is a perfect solution for exclusive client galleries!",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-password-protection/envira-password-protection.zip"
            },
            {
              "title": "Pinterest Addon",
              "slug": "envira-pinterest",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Pinterest Addon allows you to add beautiful and easy to use Pin It buttons to images in your gallery lightbox views.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-pinterest/envira-pinterest.zip"
            },
            {
              "title": "Printing Addon",
              "slug": "envira-printing",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Printing Addon for Envira Gallery provides options to output a print icon over each Gallery Image and Lightbox Image, allowing visitors to print your images.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-printing/envira-printing.zip"
            },
            {
              "title": "Proofing Addon",
              "slug": "envira-proofing",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Want to add a photo proofing process in your WordPress photo galleries? Envira adds photo proofing into WordPress to help with your client workflow.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-proofing/envira-proofing.zip"
            },
            {
              "title": "Protection Addon",
              "slug": "envira-protection",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Protection addon disables the ability for visitors to right click and download your images (both gallery display and lightbox views). It gives you complete freedom to display your work without worry.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-protection/envira-protection.zip"
            },
            {
              "title": "Slideshow Addon",
              "slug": "envira-slideshow",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Turn your gallery lightbox view into a slideshow for your visitors. You can customize settings such as autoplaying and slideshow speed.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-slideshow/envira-slideshow.zip"
            },
            {
              "title": "Social Addon",
              "slug": "envira-social",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Social Addon for Envira allows you to share your gallery images on all your favorite social networks like Facebook, Twitter and more!",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-social/envira-social.zip"
            },
            {
              "title": "Tags Addon",
              "slug": "envira-tags",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Tags addon allows you to add custom tags to each of your images in your galleries. You can then query a custom gallery from tags used for your images, and you can even do animated tag filtering within your galleries.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-tags/envira-tags.zip"
            },
            {
              "title": "Videos Addon",
              "slug": "envira-videos",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "The Videos Addon allows you to create beautiful YouTube, Vimeo, Wistia and self-hosted video galleries in Envira.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-videos/envira-videos.zip"
            },
            {
              "title": "Watermarking Addon",
              "slug": "envira-watermarking",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Protect your photos by adding custom watermarks in WordPress.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-watermarking/envira-watermarking.zip"
            },
            {
              "title": "WooCommerce Addon",
              "slug": "envira-woocommerce",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Do you want to sell your photos in WordPress? Envira gallery offers a complete integration with WooCommerce allowing you to easily sell your photos.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-woocommerce/envira-woocommerce.zip"
            },
            {
              "title": "ZIP Importer Addon",
              "slug": "envira-zip-importer",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Easily import photos from a .zip file into your gallery.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-zip-importer/envira-zip-importer.zip"
            },
            {
              "title": "Zoom Addon",
              "slug": "envira-zoom",
              "version": "69",
              "image": "http://placehold.it/400x160",
              "excerpt": "Adds zoom functionality to images in your gallery to allow up close browsing.",
              "url": "http://envira.s3.amazonaws.com/plugins/envira-zoom/envira-zoom.zip"
            }]';
            $array = json_decode($response, true);
            $response_body = json_encode($array);
        }

        // Bail out early if there are any errors.
        if ( 200 != $response_code || is_wp_error( $response_body ) ) {
            return false;
        }

        // Return the json decoded content.
        return json_decode( $response_body );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since 1.0.0
     *
     * @return object The Envira_Gallery_License object.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Envira_Gallery_License ) ) {
            self::$instance = new Envira_Gallery_License();
        }

        return self::$instance;

    }

}

// Load the license class.
$envira_gallery_license = Envira_Gallery_License::get_instance();